<?php
/**
 * PressMart functions and definitions
 *
 * @author 	PressLayouts
 * @package pressmart
 * @since 1.0
 * @version 1.0
 */

/*-----------------------------------------------------------------------*/
/* Define Constants.
/*-----------------------------------------------------------------------*/
define('PRESSMART_DIR',		get_template_directory() );			// template directory
define('PRESSMART_URI',     get_template_directory_uri() );		// template directory uri

class PressMart_Theme_Class{
	
	
	public function __construct() {
		$this->constants();
		$this->include_functions();
		add_action( 'after_setup_theme', array( $this, 'theme_setup' ), 10 );
		add_action( 'widgets_init', array( $this, 'register_sidebars' ) );
		
		if ( is_admin() ) {
			add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ) );
			add_action( 'admin_enqueue_scripts', array( $this, 'admin_style' ) );
			add_action( 'tgmpa_register', array( $this, 'register_required_plugins' ) );
		} else{		
			add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_styles' ), 100 );
			add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );		
			add_action( 'wp_enqueue_scripts', array( $this, 'dequeue_scripts' ) );		
			add_action( 'wp_head', array( $this, 'google_theme_color' ), 2 );
			add_action( 'wp_head', array( $this, 'custom_head_js') );
			add_action( 'wp_footer', array( $this, 'enqueue_inline_style'), 10 );
			add_action( 'wp_footer', array( $this, 'print_css'), 15 );
			add_action( 'wp_footer', array( $this, 'custom_footer_js') );
			add_action( 'pre_get_posts', array( $this, 'search_posts_per_page' ) );		
			add_action( 'wp', array( $this, 'post_view_count'), 999 );		
			add_filter( 'excerpt_more', array( $this, 'excerpt_more') );	
			add_filter( 'the_content_more_link',  array( $this, 'read_more_tag' ) );
			add_filter( 'excerpt_length', array( $this, 'excerpt_length'), 999 );
			add_action( 'wp_footer', array( $this, 'owl_param' ) );			
			if( PRESSMART_WOOCOMMERCE_ACTIVE ){
				add_filter( 'posts_search', array( $this, 'product_search_sku' ), 9 );
			}
		}
				
	}
	
	/**
	 * Define Constants
	 *
	 * @since   1.0
	 */
	public  function constants() {

		// Theme version
		define( 'PRESSMART_THEME_NAME', 'PressMart' );
		define( 'PRESSMART_THEME_SLUG', 'pressmart' );
		$theme = wp_get_theme( PRESSMART_THEME_SLUG );
		define( 'PRESSMART_VERSION', $theme->get( 'Version' ) );
		define( 'PRESSMART_FRAMEWORK', PRESSMART_DIR .'/inc/' );
		define( 'PRESSMART_FRAMEWORK_URI', PRESSMART_URI .'/inc/' );
		define( 'PRESSMART_ADMIN_DIR_URI', PRESSMART_FRAMEWORK_URI .'admin/' );
		define( 'PRESSMART_SCRIPTS', PRESSMART_URI .'/assets/js/' );
		define( 'PRESSMART_STYLES', PRESSMART_URI .'/assets/css/' );
		define('PRESSMART_IMAGES', PRESSMART_URI . '/assets/images/');
		define('PRESSMART_ADMIN_IMAGES', PRESSMART_ADMIN_DIR_URI . 'assets/images/');
		
		// Check if plugins are active		
		if( !defined( 'PRESSMART_WOOCOMMERCE_ACTIVE' ) ) {
			define( 'PRESSMART_WOOCOMMERCE_ACTIVE', class_exists( 'WooCommerce' ) );
		}
		if( !defined( 'PRESSMART_DOKAN_ACTIVE' ) ) {
			define( 'PRESSMART_DOKAN_ACTIVE', class_exists( 'WeDevs_Dokan' ) );
		}
		if( !defined( 'PRESSMART_WC_VENDORS_ACTIVE' ) ) {
			define( 'PRESSMART_WC_VENDORS_ACTIVE', class_exists( 'WC_Vendors' ) );
		}
		
		// Othere		
		if( !defined( 'PRESSMART_API' ) ) {
			define('PRESSMART_API', 'https://presslayouts.com/api/');
		}
		if( !defined( 'PRESSMART_PREFIX' ) ) {
			define('PRESSMART_PREFIX', '_pls_');
		}		
	}
	
	/**
	 * Load all core theme function files
	 *
	 * @since 1.0
	 */
	public function include_functions(){
		
		require_once PRESSMART_FRAMEWORK.'theme-layout.php';		
		require_once PRESSMART_FRAMEWORK.'font-config.php';
		require_once PRESSMART_FRAMEWORK.'core-functions.php';
		require_once PRESSMART_FRAMEWORK.'theme-tags.php';
		require_once PRESSMART_FRAMEWORK.'theme-functions.php';		
		require_once PRESSMART_FRAMEWORK.'theme-hooks.php';
		require_once PRESSMART_FRAMEWORK.'dynamic-css.php';
		require_once PRESSMART_FRAMEWORK.'admin/admin-function.php';	
		require_once PRESSMART_FRAMEWORK.'integrations/elementor/elementor-functions.php';	

		if( PRESSMART_WOOCOMMERCE_ACTIVE ) {
			require_once PRESSMART_FRAMEWORK.'integrations/woocommerce/wc-core-functions.php';
			require_once PRESSMART_FRAMEWORK.'integrations/woocommerce/wc-template-hooks.php';
			require_once PRESSMART_FRAMEWORK.'integrations/woocommerce/wc-template-functions.php';
			require_once PRESSMART_FRAMEWORK.'classes/class-swatches.php';
			require_once PRESSMART_FRAMEWORK.'classes/class-woocommerce.php';
			require_once PRESSMART_FRAMEWORK.'classes/class-bought-together.php';
			
			if( class_exists('WeDevs_Dokan') ){
				require_once PRESSMART_FRAMEWORK.'integrations/dokan/dokan-core-functions.php';
			}
			
			if( class_exists('WCMp') ){
				require_once PRESSMART_FRAMEWORK.'integrations/wcmp/wcmp-core-functions.php';
			}
			
			if( class_exists('WC_Vendors') ){
				require_once PRESSMART_FRAMEWORK.'integrations/wc-vendor/wc-vendors-core-functions.php';
			}
			
			if( class_exists('WCFMmp') ){
				require_once PRESSMART_FRAMEWORK.'integrations/wcfm/wcfm-core-functions.php';
			}
			
			if( function_exists( 'YITH_YWRAQ_Frontend' ) ){
				require_once PRESSMART_FRAMEWORK.'integrations/yith-add-to-quote/yith-add-to-quote-core-functions.php';
			}
		}		
		
		require_once PRESSMART_FRAMEWORK.'classes/class-metabox.php';
		require_once PRESSMART_FRAMEWORK.'classes/class-walker-nav-menu.php';
		require_once PRESSMART_FRAMEWORK.'classes/class-breadcrumb.php';
		require_once PRESSMART_FRAMEWORK.'classes/class-ajax-search.php';
		require_once PRESSMART_FRAMEWORK.'classes/sidebar-generator-class.php';
		require_once PRESSMART_FRAMEWORK.'classes/class-cookie-notice.php';
		if ( is_admin() ) {
			require_once PRESSMART_FRAMEWORK.'classes/class-tgm-plugin-activation.php';
		}
		require_once PRESSMART_FRAMEWORK.'admin/theme_options.php';
		require_once PRESSMART_FRAMEWORK.'admin/class-admin.php';
		require_once PRESSMART_FRAMEWORK.'admin/class-dashboard.php';
		require_once PRESSMART_FRAMEWORK.'admin/class-update-theme.php';
	}
	
	/**
	 * Theme Setup
	 *
	 * @since   1.0
	 */
	public function theme_setup() {
	
		load_theme_textdomain( 'pressmart', get_template_directory() . '/languages' );	
		load_theme_textdomain( 'pressmart', get_stylesheet_directory() . '/languages' );
		
		/* Theme support */
		add_theme_support( 'automatic-feed-links' );
		add_theme_support( 'title-tag' );
		add_theme_support( 'post-thumbnails' );	
		add_theme_support( 'post-formats', array( 'image', 'gallery', 'video', 'audio', 'quote', 'link' ) );
		add_theme_support( 'html5', array( 'gallery', 'caption' ) );
		add_theme_support( 'wp-block-styles' );
				
		// Add support for responsive embedded content.
        add_theme_support( 'responsive-embeds' );		
		
		// Disable Widget block editor.
		if( apply_filters('pressmart_disable_widgets_block_editor', true) ) {
			remove_theme_support( 'block-templates' );
			remove_theme_support( 'widgets-block-editor' );
		}
		
		// Set the default content width.
		$GLOBALS['content_width'] = 1200;
		
		register_nav_menus( array(
			'primary' 					=> esc_html__( 'Primary Menu', 'pressmart' ),
			'secondary'					=> esc_html__( 'Secondary Menu', 'pressmart' ),
			'categories-menu' 			=> esc_html__( 'Categories(Vertical) Menu', 'pressmart' ),
			'topbar-menu' 				=> esc_html__( 'Topbar Menu', 'pressmart' ),
			'mobile-menu' 				=> esc_html__( 'Mobile Menu', 'pressmart' ),
			'mobile-categories-menu' 	=> esc_html__( 'Mobile Categories Menu', 'pressmart' ),
			'myaccount-menu' 			=> esc_html__( 'MyAccount/Profile Menu', 'pressmart' ),
		) );
	}
	
	/*-----------------------------------------------------------------------*/
	/* Register custom fonts.
	/*-----------------------------------------------------------------------*/
	public function fonts_url() {
		$fonts_url = '';	
		
		if ( 'off' !== _x( 'on', 'Poppins font: on or off', 'pressmart' ) ) {
			$font_families[] = 'Poppins:300,400,500,600,700,900';
		}

		if ( ! empty( $font_families ) ) {
			$query_args = array(
				'family' => urlencode( implode( '|', $font_families ) ),
				'subset' => urlencode( 'latin,latin-ext' ),
			);

			$fonts_url = add_query_arg( $query_args, '//fonts.googleapis.com/css' );
		}

		return esc_url_raw( $fonts_url );		
	}
	
	/**
	 * Register required plugins
	 *
	 * @since   1.0
	*/
	public function register_required_plugins(){
		$plugins = array(
			array(
				'name' 					=> 'PressMart Core',
				'slug' 					=> 'pressmart-core',
				'source' 				=> $this->pressmart_get_tgm_plugin_path( 'pressmart-core.zip' ),
				'version'  				=> '1.6.5',
				'required' 				=> true,
			),
			array(
				'name' 					=> 'Revolution Slider',
				'slug' 					=> 'revslider',
				'source'             	=> $this->pressmart_get_tgm_plugin_path( 'revslider.zip' ),
				'version'  				=> '6.7.18',
				'required' 				=> true,
			),
			array(
				'name' 					=> 'Elementor Website Builder',
				'slug' 					=> 'elementor',
				'required' 				=> true,
			),		
			array(
				'name' 					=> 'Woocommerce',
				'slug' 					=> 'woocommerce',
				'required' 				=> true,
			),
			array(
				'name' 					=> 'YITH WooCommerce Compare',
				'slug' 					=> 'yith-woocommerce-compare',
				'required' 				=> false,
			),
			array(
				'name' 					=> 'YITH WooCommerce Wishlist',
				'slug' 					=> 'yith-woocommerce-wishlist',
				'required' 				=> false,
			),	
			array(
				'name' 					=> 'MailChimp for WordPress',
				'slug' 					=> 'mailchimp-for-wp',
				'required' 				=> false,
			),
			array(
				'name'      			=> 'Contact Form 7',
				'slug'     			 	=> 'contact-form-7',
				'required' 			 	=> false,
			),
		);		
		$config = array(
			'id'           => 'tgmpa',
			'menu'         => 'pls-install-plugins',
			'parent_slug'  => 'themes.php',
			'capability'   => 'edit_theme_options',
			'has_notices'  => true,
			'dismissable'  => true,
			'is_automatic' => false,			
		);
		tgmpa( $plugins, $config );
	}
	
	public function pressmart_get_tgm_plugin_path($plugin_name = ''){
	
		$is_license_activated = ( get_option( 'pressmart_is_activated' ) && get_option( 'envato_purchase_code_39241221' ) ) ? true : false;
		
		// bail early if no plugin name provided
		if( empty($plugin_name) ) return '';
		if( !$is_license_activated ) return '';
		$plugin_url = get_option( 'pressmart_plugin_file' );
		return $plugin_url.$plugin_name;
	}
	
	/**
	 * Registers sidebars
	 *
	 * @since   1.0
	 */
	public function register_sidebars(){

		register_sidebar( array(
			'name'          => esc_html__( 'Blog Sidebar', 'pressmart' ),
			'id'            => 'sidebar-1',
			'description'   => esc_html__( 'Add widgets here to appear in your sidebar.', 'pressmart' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		) );
		register_sidebar( array(
			'name'          => esc_html__( 'Shop Page Sidebar', 'pressmart' ),
			'id'            => 'shop-page',
			'description'   => esc_html__( 'Add widgets here to appear in shop page sidebar.', 'pressmart' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		) );
		register_sidebar( array(
			'name'          => esc_html__( 'Shop Filter Sidebar', 'pressmart' ),
			'id'            => 'shop-filters-sidebar',
			'description'   => esc_html__( 'Add widgets here to appear in your shop page.', 'pressmart' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h2 class="widget-title">',
			'after_title'   => '</h2>',
		) );
		register_sidebar( array(
			'name'          => esc_html__( 'Product Page Sidebar', 'pressmart' ),
			'id'            => 'single-product',
			'description'   => esc_html__( 'Add widgets here to appear in single product page sidebar.', 'pressmart' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		) );
		register_sidebar( array(
			'name'          => esc_html__( 'Footer Area 1', 'pressmart' ),
			'id'            => 'footer-widget-area-1',
			'description'   => esc_html__( 'Add widgets here to appear in your footer.', 'pressmart' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		) );
		register_sidebar( array(
			'name'          => esc_html__( 'Footer Area 2', 'pressmart' ),
			'id'            => 'footer-widget-area-2',
			'description'   => esc_html__( 'Add widgets here to appear in your footer.', 'pressmart' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		) );
		register_sidebar( array(
			'name'          => esc_html__( 'Footer Area 3', 'pressmart' ),
			'id'            => 'footer-widget-area-3',
			'description'   => esc_html__( 'Add widgets here to appear in your footer.', 'pressmart' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		) );
		register_sidebar( array(
			'name'          => esc_html__( 'Footer Area 4', 'pressmart' ),
			'id'            => 'footer-widget-area-4',
			'description'   => esc_html__( 'Add widgets here to appear in your footer.', 'pressmart' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		) );
		register_sidebar( array(
			'name'          => esc_html__( 'Footer Area 5', 'pressmart' ),
			'id'            => 'footer-widget-area-5',
			'description'   => esc_html__( 'Add widgets here to appear in your footer.', 'pressmart' ),
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		) );
	}
	
	/**
	 * Load scripts in the WP admin
	 *
	 * @since 1.0
	 */
	public function admin_style( $hook ) {
		//Admin css
		global $pagenow,$typenow;
		wp_enqueue_style( 'wp-color-picker' );		
		wp_enqueue_style( 'presslayouts-font', PRESSMART_STYLES.'presslayouts-font.css', array(), '1.0' );
		
		if ( strpos( $hook, 'pressmart-demo-import' ) !== false ){
			wp_enqueue_style( 'magnific-popup', PRESSMART_STYLES.'magnific-popup.css', array(), PRESSMART_VERSION );
		}
		
		if ( 'customize.php' != $pagenow ) {
			wp_enqueue_style( 'pressmart-style', PRESSMART_FRAMEWORK_URI.'admin/assets/css/admin.css', array(), PRESSMART_VERSION );
		}
		
		if( $typenow == 'pls_size_chart' ){
			wp_register_style( 'pressmart-edittable', PRESSMART_FRAMEWORK_URI.'admin/assets/css/jquery.edittable.css', null, time() );
			wp_enqueue_style('pressmart-edittable');
		}
		
	}
	
	/**
	 * Load scripts in the WP admin
	 *
	 * @since 1.0
	 */
	public function admin_scripts( $hook ) {
		global $pagenow, $typenow;
		wp_enqueue_media(); 
		wp_enqueue_script( 'wp-color-picker' );
				
		if ( 'toplevel_page_pressmart-theme' == $hook ) {
			wp_enqueue_script( 'pressmart-activation-theme', PRESSMART_FRAMEWORK_URI.'admin/assets/js/theme-activation.js');
		}
		
		if ( strpos( $hook, 'pls-system-status' ) !== false ){
			wp_enqueue_script( 'pls-system-status', PLS_FRAMEWORK_URI.'admin/assets/js/system-status.js' );
		}
		
		if ( 'nav-menus.php' == $pagenow ) {
			wp_enqueue_style( 'presslayouts-font', PRESSMART_STYLES.'presslayouts-font.css', array(), '1.0' );
			wp_enqueue_script( 'pressmart-mega-menu', PRESSMART_FRAMEWORK_URI.'admin/assets/js/mega-menu.js');
		}
		
		if( $typenow == 'pls_size_chart' ){
			wp_register_script( 'pressmart-edittablejs', PRESSMART_FRAMEWORK_URI.'admin/assets/js/jquery.edittable.js', array('jquery'), time(), true );
			wp_enqueue_script('pressmart-edittablejs');
		}
		
		if ( strpos( $hook, 'pressmart-demo-import' ) !== false ){
			wp_enqueue_script( 'magnific-popup', PRESSMART_SCRIPTS.'jquery.magnific-popup.min.js', array(), PRESSMART_VERSION );
		}
		
		wp_enqueue_script( 'pressmart-admin-js', PRESSMART_FRAMEWORK_URI.'admin/assets/js/admin.js' );
		wp_localize_script( 'pressmart-admin-js', 'pressmart_admin_params', apply_filters('pressmart_admin_js_params', array(
			'ajaxurl'          		=> admin_url( 'admin-ajax.php' ),
			'nonce'            		=> wp_create_nonce( 'pressmart_nonce' ),
			'loading_text'      	=> esc_html__( 'Loading...', 'pressmart' ),
			'bindmessage'      		=> esc_html__( 'Are you sure you want to leave?','pressmart' ),
			'demo_success'      	=> esc_html__( 'Demo imported successfully.', 'pressmart' ),
			'menu_icon_change_text'	=> esc_html__( 'Change Custom Icon', 'pressmart' ),
			'menu_icon_upload_text'	=> esc_html__( 'Upload Custom Icon', 'pressmart' ),
			'menu_delete_icon_msg'	=> esc_html__( 'Are you sure,You want to remove this icon?', 'pressmart' ),
		)));
	}

	/**
	 * Disable Unused Scripts
	 */
	function dequeue_scripts() {
		
		// Disable font awesome style from plugins
		if ( pressmart_get_option( 'disable-fontawesome', 1 ) ) {
			wp_deregister_style( 'fontawesome' );
			wp_deregister_style( 'font-awesome' );
			wp_deregister_style( 'wplc-font-awesome' );
		}
		
		// YITH WCWL styles & scripts
		if ( defined( 'YITH_WCWL' ) && ! defined( 'YITH_WCWL_PREMIUM' ) ) {
			
			wp_dequeue_style( 'jquery-selectBox' );
			wp_dequeue_script( 'jquery-selectBox' );
			
			// checkout
			if ( function_exists( 'is_checkout' ) && is_checkout() ) {
				wp_dequeue_style( 'selectWoo' );
				wp_deregister_style( 'selectWoo' );
			}
		}
		
		if ( function_exists( 'yith_wcwl_is_wishlist_page' ) && !yith_wcwl_is_wishlist_page() ) {
			// YITH : main style was dequeued because of font-awesome
			wp_dequeue_style( 'yith-wcwl-main' );
			wp_dequeue_style( 'yith-wcwl-font-awesome' );
			wp_deregister_style( 'woocommerce_prettyPhoto_css' );
		}
		
		// WooCommerce PrettyPhoto(deprecated), but YITH Wishlist use
		if ( class_exists( 'WooCommerce' ) && ! defined( 'YITH_WCWL_PREMIUM' ) ) {
			wp_dequeue_style( 'woocommerce_prettyPhoto_css' );			
			wp_dequeue_script( 'prettyPhoto-init' );
			wp_dequeue_script( 'prettyPhoto' );
		}

		// Disable wp block library 
		if ( pressmart_get_option( 'disable-gutenberg', 0 ) ) {
			wp_deregister_style( 'wp-block-library' );
			wp_dequeue_style( 'wp-block-library' );
		}
		
		// Disable Default wc blocks styles
		if ( pressmart_get_option( 'disable-wc-blocks', 0 ) ) {
			//wp_dequeue_style( 'wc-blocks-style' );
			//wp_deregister_style( 'wc-blocks-style' );
			wp_dequeue_style( 'wc-blocks-vendors-style' );
			wp_deregister_style( 'wc-blocks-vendors-style' );
		}
		
		// REMOVE WP EMOJI
		remove_action( 'wp_head', 'print_emoji_detection_script', 7 );
		remove_action( 'wp_print_styles', 'print_emoji_styles');
	}

	/**
	 * Load front-end css
	 *
	 * @since 1.0
	 */
	public function enqueue_styles() {
				
		// Load our main stylesheet.
		wp_enqueue_style( 'pressmart-style', PRESSMART_URI.'/style.css' , array(), PRESSMART_VERSION );
		
		// Load elementor css
		wp_enqueue_style( 'elementor-frontend' );

		$theme = ( is_rtl() ) ? PRESSMART_STYLES .'theme-rtl' : PRESSMART_STYLES .'theme';
		$woocommerce_style = ( is_rtl() ) ? PRESSMART_STYLES .'woocommerce-rtl' : PRESSMART_STYLES .'woocommerce' ;		
		
		if ( ! class_exists('ReduxFrameworkPlugin') ) {
			wp_enqueue_style( 'pressmart-fonts', $this->fonts_url(), array(), null );
		}
		wp_enqueue_style( 'bootstrap-grid', PRESSMART_STYLES.'bootstrap-grid.css', array(), '4.1.3' );
		wp_enqueue_style( 'presslayouts-font', PRESSMART_STYLES.'presslayouts-font.css', array(), '1.0' );
		wp_enqueue_style( 'pressmart-woocommerce', $woocommerce_style.'.css' , array(), '' );
		wp_enqueue_style( 'owl-carousel', PRESSMART_STYLES.'owl.carousel.min.css', array(), '2.3.4' );
		wp_enqueue_style( 'slick', PRESSMART_STYLES . 'slick.css', array(), '1.9.0' );
		wp_enqueue_style( 'magnific-popup', PRESSMART_STYLES.'magnific-popup.css', array(), '1.1.0' );
		wp_enqueue_style( 'animate', PRESSMART_STYLES.'animate.min.css', array(), '3.7.2' );	
				
		// Theme basic stylesheet.
		wp_enqueue_style( 'pressmart-base', $theme.'.css', array( 'bootstrap-grid', 'pressmart-woocommerce' ), PRESSMART_VERSION );
		
		//Dynamic CSS
		wp_add_inline_style( 'pressmart-base', pressmart_theme_style() );
		
		// load typekit fonts
		$enable_typekit_font 	= pressmart_get_option( 'typekit-font', 0 );
		$typekit_id 			= pressmart_get_option( 'typekit-kit-id', '' );

		if ( $enable_typekit_font && !empty($typekit_id)) {
			wp_enqueue_style( 'pressmart-typekit',  pressmart_get_protocol().'//use.typekit.net/' . esc_attr ( $typekit_id ) . '.css', array(), PRESSMART_VERSION );
		}
		
		wp_register_style( 'pressmart-custom-css', false );
	}
	
	/**
	 * Load front-end script
	 *
	 * @since 1.0
	 */
	public function enqueue_scripts() {
		
		// Load waypoints Js
		wp_enqueue_script( 'waypoints', PRESSMART_SCRIPTS .'waypoints.min.js', array( 'jquery' ), '2.0.2', true );
		wp_enqueue_script( 'popper', PRESSMART_SCRIPTS.'popper.min.js', array( 'jquery' ), '4.0.0', true );
		wp_enqueue_script( 'bootstrap', PRESSMART_SCRIPTS.'bootstrap.min.js', array( 'jquery' ), '4.3.1', true );
		wp_enqueue_script( 'owl-carousel', PRESSMART_SCRIPTS.'owl.carousel.min.js', array( 'jquery' ), '2.3.4', true );
		wp_register_script( 'isinviewport', PRESSMART_SCRIPTS.'isInViewport.min.js', array( 'jquery' ), '1.8.0', true );
		wp_enqueue_script( 'slick', PRESSMART_SCRIPTS.'slick.min.js', array( 'jquery' ), '1.9.0', true );
		wp_register_script( 'isotope', PRESSMART_SCRIPTS.'isotope.pkgd.min.js', array( 'jquery' ), '3.0.6', true );
		wp_register_script( 'cookie', PRESSMART_SCRIPTS.'cookie.min.js', array( 'jquery' ), '', true );
		wp_register_script( 'parallax', PRESSMART_SCRIPTS.'jquery.parallax.js', array( 'jquery' ), '', true );
		wp_register_script( 'threesixty', PRESSMART_SCRIPTS .'threesixty.min.js', array( 'jquery' ), '2.0.5', true );
		wp_enqueue_script ( 'magnific-popup', PRESSMART_SCRIPTS.'jquery.magnific-popup.min.js', array( 'jquery' ), '1.1.0', true );
		wp_enqueue_script( 'nanoscroller', PRESSMART_SCRIPTS.'jquery.nanoscroller.min.js', array( 'jquery' ), '0.8.7', true );
		wp_register_script( 'countdown', PRESSMART_SCRIPTS.'jquery.countdown.min.js', array( 'jquery' ), '2.2.0', true );
		wp_register_script( 'counterup', PRESSMART_SCRIPTS.'jquery.counterup.min.js', array( 'jquery' ), '1.0', true );
		wp_register_script( 'sticky-kit', PRESSMART_SCRIPTS.'sticky-kit.min.js', array( 'jquery' ), '1.10.0', true );
		if( pressmart_get_option( 'product-ajax-search', 1 ) ){
			wp_enqueue_script( 'pressmart-autocomplete', PRESSMART_SCRIPTS.'jquery.autocomplete.min.js', array( 'jquery' ), '', true );
		}
		if( pressmart_get_option( 'lazy-load', 0 ) ){
			wp_enqueue_script( 'lazyload', PRESSMART_SCRIPTS .'jquery.lazy.min.js', array( 'jquery' ), PRESSMART_VERSION, true );
		}
		if( pressmart_get_option( 'widget-items-hide-max-limit', 0 ) ){
			wp_enqueue_script( 'hideMaxListItem', PRESSMART_SCRIPTS.'hideMaxListItem-min.js', array( 'jquery' ), '1.36', true );
		}	
		if( class_exists( 'WooCommerce' ) && pressmart_get_option( 'product-quickview-button', 1 ) ){
			wp_enqueue_script( 'wc-add-to-cart-variation' );
		}
		
		if ( ! wp_script_is( 'wc-cart-fragments', 'enqueued' ) && wp_script_is( 'wc-cart-fragments', 'registered' ) ) {
			wp_enqueue_script( 'wc-cart-fragments' );
		}
	
		if( pressmart_get_option( 'sticky-sidebar', 1 ) && ( 'full-width' != pressmart_get_layout() ) ){
			wp_enqueue_script( 'sticky-kit' );
		}
		if ( function_exists('is_product') && is_product() && ( pressmart_get_option( 'sticky-product-image', 1 ) || pressmart_get_option( 'sticky-product-summary', 1 ) ) ){
			wp_enqueue_script( 'sticky-kit' );			
		}		
		
		$google_api_key = pressmart_get_option( 'google-map-api', '' );
		if( ! empty( $google_api_key ) ){
			wp_enqueue_script( 'pressmart-google-map-api', pressmart_get_protocol().'//maps.google.com/maps/api/js?sensor=false&libraries=geometry&v=3.22&key=' . $google_api_key . '', array(), '', false );
		}		
		wp_enqueue_script( 'pressmart-script', PRESSMART_SCRIPTS.'functions.js', array( 'jquery' ), PRESSMART_VERSION, true );
		
		$is_rtl = is_rtl() ? true : false ;		
		$pressmart_settings = apply_filters( 'pressmart_localize_script_data', array(
			'rtl' 							=> $is_rtl,
			'ajax_url' 						=> admin_url( 'admin-ajax.php' ),
			'ajax_nonce' 					=> esc_js( wp_create_nonce( 'pressmart-ajax-nonce' ) ),
			'product_ajax_search'			=> pressmart_get_option( 'product-ajax-search', 1 ) ? true : false,	
			'sticky_header'					=> pressmart_get_option( 'header-sticky', 0 ) ? true : false,
			'sticky_header_class'			=> pressmart_get_option( 'header-sticky-part', 'main' ),
			'sticky_header_scroll_up'		=> pressmart_get_option( 'header-sticky-scroll-up', 0 ) ? true : false,	
			'sticky_header_tablet'			=> pressmart_get_option( 'header-sticky-tablet', 0 ) ? true : false,	
			'sticky_header_mobile'			=> pressmart_get_option( 'header-sticky-mobile', 0 ) ? true : false,
			'login_register_popup'			=> pressmart_get_option( 'login-register-popup', 1 ) ? true : false,
			'button_loader'					=> true,
			'header_minicart_popup'			=> pressmart_get_option( 'header-minicart-popup', 'slider' ),
			'promo_bar'						=> pressmart_get_option( 'promo-bar', 0 ) ? true : false,	
			'lazy_load'						=> pressmart_get_option( 'lazy-load', 0 ) ? true : false,	
			'cookie_path'					=> COOKIEPATH,
			'cookie_expire'					=> 3600 * 24 * 30,			
			'show_promobar_in_next_days'	=> 1,			
			'permalink'						=> ( get_option( 'permalink_structure' ) == '' ) ? 'plain' : '',			
			'newsletter_args'				=> apply_filters( 'pressmart_js_newsletter_args', array(
				'newsletter_popup'			=> pressmart_get_option( 'newsletter-popup', 0 ) ? true : false,
				'popup_display_on'		=> pressmart_get_option( 'newsletter-when-appear', 'page_load' ),
				'popup_delay'			=> pressmart_get_option( 'newsletter-delay', 5 ),
				'popup_x_scroll'		=> pressmart_get_option( 'newsletter-x-scroll', 30 ),
				'show_for_mobile'		=> pressmart_get_option( 'newsletter-show-mobile', 1 ),
				'show_in_next_days'		=> 1,
				'version'				=> pressmart_get_option( 'newsletter-version', 1 ),
			) ),
			'js_translate_text'				=> apply_filters( 'pressmart_js_text', array(
				'days_text'					=> esc_html__( 'Days', 'pressmart' ),
				'hours_text'				=> esc_html__( 'Hours', 'pressmart' ),
				'mins_text'					=> esc_html__( 'Mins', 'pressmart' ),
				'secs_text'					=> esc_html__( 'Secs', 'pressmart' ),
				'sdays_text'				=> esc_html__( 'd', 'pressmart' ),
				'shours_text'				=> esc_html__( 'h', 'pressmart' ),
				'smins_text'				=> esc_html__( 'm', 'pressmart' ),
				'ssecs_text'				=> esc_html__( 's', 'pressmart' ),
				'show_more'					=> esc_html__( '+ Show more', 'pressmart' ),
				'show_less'					=> esc_html__( '- Show less', 'pressmart' ),
				'loading_txt'				=> esc_html__( 'Loading...', 'pressmart' ),
				'variation_unavailable'		=> esc_html__( 'Sorry, this product is unavailable. Please choose a different combination.', 'pressmart' ),
			) ),
			'cart_auto_update'				=> pressmart_get_option( 'cart-auto-update', 1 ) ? true : false,
			'checkout_product_quantity'		=> pressmart_get_option( 'checkout-product-quantity', 0 ) ? true : false,
			'product_image_zoom'			=> pressmart_get_option( 'product-gallery-zoom', 1 ) ? true : false,
			'product_PhotoSwipe'			=> pressmart_get_option( 'product-gallery-lightbox', 1 ) ? true : false,
			'product_gallery_layout'		=> function_exists('pressmart_get_product_gallery_layout') ? pressmart_get_product_gallery_layout() : pressmart_get_option( 'product-gallery-style', 'product-gallery-left' ),
			'typeahead_options'     		=> array( 'hint' => false, 'highlight' => true ),
			'nonce'                 		=> wp_create_nonce( 'pressmart_nonce' ),						
			'enable_add_to_cart_ajax' 		=> pressmart_get_option('product-add-to-cart-ajax', 1 ) ? true : false,
			'mini_cart_popup' 				=> ( 'slider' == pressmart_get_option( 'header-minicart-popup', 'slider' ) )  ? true : false,
			'sticky_product_image'			=> pressmart_get_option( 'sticky-product-image', 1 ) ? true : false,
			'sticky_product_summary'		=> pressmart_get_option( 'sticky-product-summary', 1 ) ? true : false,
			'sticky_sidebar'				=> pressmart_get_option( 'sticky-sidebar', 1 ) ? ( ( pressmart_is_catalog() && pressmart_get_option( 'shop-page-off-canvas-sidebar', 0 ) ) ? false : true) : false,
			'widget_toggle'					=> pressmart_get_option('widget-toggle', 0 ) ? true : false,
			'widget_menu_toggle'			=> pressmart_get_option('widget-menu-toggle', 0 ) ? true : false,
			'widget_hide_max_limit_item' 	=> pressmart_get_option('widget-items-hide-max-limit', 0 ) ? true : false,
			'number_of_show_widget_items'	=> pressmart_get_option('number-of-show-widget-items', 8 ),			
			'touch_slider_mobile'			=> pressmart_get_option('touch-slider-mobile', 0 ) ? true : false,
			'disable_variation_price_change'=> false,
			'maintenance_mode'				=> pressmart_get_option( 'maintenance-mode' , 0 ) ? true : false,
			
		));
		
		if ( class_exists( 'WooCommerce' ) ) {
			$pressmart_settings['price_format']             = get_woocommerce_price_format();
			$pressmart_settings['price_decimals']           = wc_get_price_decimals();
			$pressmart_settings['price_thousand_separator'] = wc_get_price_thousand_separator();
			$pressmart_settings['price_decimal_separator']  = wc_get_price_decimal_separator();
			$pressmart_settings['currency_symbol']          = get_woocommerce_currency_symbol();
			$pressmart_settings['wc_tax_enabled']           = wc_tax_enabled();
			$pressmart_settings['cart_url']                 = wc_get_cart_url();
			if ( wc_tax_enabled() ) {
				$pressmart_settings['ex_tax_or_vat'] = WC()->countries->ex_tax_or_vat();
			} else {
				$pressmart_settings['ex_tax_or_vat'] = '';
			}
		}
		
		//general ajax
		wp_localize_script( 'pressmart-script', 'pressmart_options', $pressmart_settings );			
		
		wp_enqueue_script( 'html5', PRESSMART_SCRIPTS .'html5.js' , array(), '3.7.3' );
		wp_script_add_data( 'html5', 'conditional', 'lt IE 9' );
		
		if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
			wp_enqueue_script( 'comment-reply' );
		}
	}	

	/**
	 * Load custom js in footer
	 * @since 1.0
	 */
	function owl_param() {
		global $pressmart_owlparam;
		wp_localize_script( 'pressmart-script', 'pressmartOwlArg', (array) $pressmart_owlparam );
	}
	
	/**
	 * Search product with sku
	 * @since 1.0
	 */
	public function product_search_sku( $where ) {
        global $pagenow, $wpdb, $wp;
 
        if ( ( is_admin() && 'edit.php' != $pagenow )
             || ! is_search()
             || ! isset( $wp->query_vars['s'] )
             || ( isset( $wp->query_vars['post_type'] ) && 'product' != $wp->query_vars['post_type'] )
             || ( isset( $wp->query_vars['post_type'] ) && is_array( $wp->query_vars['post_type'] ) && ! in_array( 'product', $wp->query_vars['post_type'] ) )
        ) {
            return $where;
        }
        $search_ids = array();
        $terms      = explode( ',', $wp->query_vars['s'] );
 
        foreach ( $terms as $term ) {
            //Include the search by id if admin area.
            if ( is_admin() && is_numeric( $term ) ) {
                $search_ids[] = $term;
            }
            // search for variations with a matching sku.
 
            $sku_to_parent_id = $wpdb->get_col( $wpdb->prepare( "SELECT p.post_parent as post_id FROM {$wpdb->posts} as p join {$wpdb->postmeta} pm on p.ID = pm.post_id and pm.meta_key='_sku' and pm.meta_value LIKE '%%%s%%' where p.post_parent <> 0 group by p.post_parent", wc_clean( $term ) ) );
 
            //Search for a simple product that matches the sku.
            $sku_to_id = $wpdb->get_col( $wpdb->prepare( "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key='_sku' AND meta_value LIKE '%%%s%%';", wc_clean( $term ) ) );
 
            $search_ids = array_merge( $search_ids, $sku_to_id, $sku_to_parent_id );
        }
 
        $search_ids = array_filter( array_map( 'absint', $search_ids ) );
 
        if ( sizeof( $search_ids ) > 0 ) {
            $where = str_replace( ')))', ") OR ({$wpdb->posts}.ID IN (" . implode( ',', $search_ids ) . "))))", $where );
        }
 
        return $where;
    }
		
	function google_theme_color(){
		
		$google_theme_color = pressmart_get_option( 'google-theme-color', 'transparent' );
		
		if( 'transparent' != $google_theme_color){ ?>	
			<meta name="theme-color" content="<?php echo esc_attr( $google_theme_color ); ?>">
		<?php
		}
	}
	
	/**
	 * Output of custom js options.
	 */
	public function custom_head_js() {
		
		$custom_js = pressmart_get_option('custom-js-header','');
		
		if ( !empty( trim( $custom_js ) ) ) {
			
			echo apply_filters( 'pressmart_head_custom_js', $custom_js ); // WPCS: XSS OK.
			
		}
	}

	/**
	* Javascript detection
	*/
	public function custom_footer_js(){
		
		$custom_js 	= trim( pressmart_get_option('custom_js', '') );
		$output = '';
		
		if( !empty( $custom_js ) ){ 
			$output .= '<script>' ;
			$output .= $custom_js ;
			$output .= '</script>' ;
		}
		echo apply_filters( 'pressmart_custom_js', $output ); // WPCS: XSS OK.
	}
	
	/**
	 * Output of dyanamic css.
	 */
	public  function print_css() {
		global $pressmart_custom_css;

		if ( ! empty( trim( (string)$pressmart_custom_css ) ) ) {
			// Sanitize.
			$pressmart_custom_css = wp_check_invalid_utf8( $pressmart_custom_css );			
			wp_add_inline_style( 'pressmart-custom-css', $pressmart_custom_css );
		}
	}
	
	/**
	 * Enqueue custom inline style
	 */
	public function enqueue_inline_style(){
		wp_enqueue_style( 'pressmart-custom-css' );
	}
	
	/**
	 * Alter the search posts per page
	 *
	 * @since 1.0
	 */
	public  function search_posts_per_page( $query ) {
		
		if ( is_admin() || ! $query->is_main_query() ) return;
		$portfolio_per_page = pressmart_get_option( 'portfolio-per-page' );
		
		if ( in_array ( $query->get('post_type'), array('portfolio') ) ) {
			$query->set( 'posts_per_page', $portfolio_per_page);
			return;
		}elseif( $query->is_main_query() && is_search() && isset( $_GET['post_type'] ) && $_GET['post_type'] == 'product' ){
			$posts_per_page = pressmart_get_option( 'products-per-page', 12);
			if ( isset( $_GET[ 'per_page' ] ) ) {
				$posts_per_page = $_GET[ 'per_page' ];
			}
			$query->set( 'posts_per_page', $posts_per_page);
		}
	}
	
	/**
	 *Post View Count 
	 */
	public function post_view_count(){
		$prefix = PRESSMART_PREFIX;
		if( ! is_single() || ! is_singular( 'post' ) ) return;
		$post_id = get_the_ID();
		$views = get_post_meta( $post_id, $prefix.'views_count', true );
		$views = !empty($views) ? $views : 0;
		
		update_post_meta( $post_id, $prefix.'views_count', ($views+1) );
		$views = get_post_meta( $post_id, $prefix.'views_count', true );
	}
	
	
	/**
	 * 'Continue reading' link.
	 */
	public function excerpt_more( $link ) {
		return '';
	}
	
	public function read_more_tag() {
		
		return sprintf( '<p class="read-more-btn link-more"><a href="%1$s" class="more-link">%2$s</a></p>',
			esc_url( get_permalink( get_the_ID() ) ),
			pressmart_get_loop_prop( 'post-readmore-text' )
		);
	}

	/**
	 * Filter the except length to 30 words.
	 */
	function excerpt_length( $length ) {
		return pressmart_get_option( 'blog-excerpt-length', 30 );
	}
} 
// Initialize theme
$pressmart_theme_class = new PressMart_Theme_Class;